/* ******************************** 
 *  library : TM1637Full.h
 *  
 *  Gestion d'un circuit TM1637 pour piloter un tube d'afficheurs 7 segments
 *  4 ou 6 digits ainsi que d'un clavier 6 touches.
 *  Projet réalisé sur les bases de la bibliothèque TM1637Display de <avishorp@gmail.com>
 *  
 *  @Author mrcjl <mjl1@morceau.fr>
 *  @Version 2.3 (23/12/2018)
 *  @see http://wikitechno.morceau.fr/doku.php?id=mbloc_disp4x7seg
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
 
extern "C" {
  #include <stdlib.h>
  #include <string.h>
  #include <inttypes.h>
}

#include "TM1637Full.h"
#include <Arduino.h>

#define TM1637_I2C_COMM1    0x40
#define TM1637_I2C_COMM2    0xC0
#define TM1637_I2C_COMM3    0x80
#define TM1637_I2C_COMM4    0x42

//
//      A
//     ---
//  F |   | B
//     -G-
//  E |   | C
//     ---
//      D
const uint8_t digitToSegment[] = {
 // XGFEDCBA
  0b00111111,    // 0
  0b00000110,    // 1
  0b01011011,    // 2
  0b01001111,    // 3
  0b01100110,    // 4
  0b01101101,    // 5
  0b01111101,    // 6
  0b00000111,    // 7
  0b01111111,    // 8
  0b01101111,    // 9
  0b01110111,    // A
  0b01111100,    // b
  0b00111001,    // C
  0b01011110,    // d
  0b01111001,    // E
  0b01110001,    // F
  0b01000000,    // -
  0b00000001,    // haut
  0b00000010,    // hautdroite
  0b00000100,    // basdroite
  0b00001000,    // bas
  0b00010000,    // basgauche
  0b00100000,    // hautgauche
  0b01010010,    // monte
  0b01100100,    // descend
  0b01110110,    // H
  0b01100011,
  0b01011100,
  0b01001000,
  0b01001001,
  0b01000001
  };


TM1637Full::TM1637Full(uint8_t pinDIO, uint8_t pinClk, uint8_t nbDigit) {
  // Copie les paramètres
  m_pinClk = pinClk;
  m_pinDIO = pinDIO;
  if (nbDigit > 6) nbDigit = 6;
  m_nbDigit = nbDigit;

  // Set the pin direction and default value.
  // Both pins are set as inputs, allowing the pull-up resistors to pull them up
  pinMode(m_pinClk, INPUT);
  pinMode(m_pinDIO,INPUT);
  digitalWrite(m_pinClk, LOW);
  digitalWrite(m_pinDIO, LOW);
}

void TM1637Full::setBrightness(uint8_t brightness, bool on) {
  m_brightness = (brightness & 0x7) | (on? 0x08 : 0x00);
}

void TM1637Full::setSegments(const uint8_t segments[], uint8_t length, uint8_t pos) {
  // Write COMM1
  start();
  writeByte(TM1637_I2C_COMM1);
  stop();

  // Write COMM2 + first digit address
  start();
  writeByte(TM1637_I2C_COMM2 + (pos & 0x07));

  // Write the data bytes
  for (uint8_t k=0; k < m_nbDigit; k++) {
    if (k > length) writeByte(0);
    else writeByte(segments[k]);
  }
  stop();

  // Write COMM3 + brightness
  start();
  writeByte(TM1637_I2C_COMM3 + (m_brightness & 0x0f));
  stop();
}

uint8_t TM1637Full::getBtn() {
  uint8_t val;
  // Write COMM4
  start();
  writeByte(TM1637_I2C_COMM4);
  
  // Read
  uint8_t code = readByte();
  switch (code) {
    case 255: val = 0; break;
    case 239: val = 1; break;
    case 111: val = 2; break;
    case 175: val = 3; break;
    case 47: val = 4; break;
    case 207: val = 5; break;
    case 79: val = 6; break;
    default: val = 255;
  }
  return val;
}

void TM1637Full::showRaw(const uint8_t segments[]) {
  uint8_t digits[6];
  for (uint8_t i = 0; i<6; i++) {
    if (segments[i] & 0x80) digits[i] = encodeDigit(segments[i] & 0x7F, true);
    else digits[i] = encodeDigit(segments[i], false);
  }
  setSegments(digits + (6 - m_nbDigit), m_nbDigit);
}

void TM1637Full::showNumberDec(long num, bool leading_zero, uint8_t decimal) {
  uint8_t digits[6];
  const static long divisors[] = { 1, 10, 100, 1000, 10000, 100000 };
  bool leading = true;
  uint8_t dots = 0x4 << decimal;

  for(int8_t k = 0; k < 6; k++) {
    // Calcule la valeur du digit
    long divisor = divisors[5 - k];
    int d = num / divisor;
    uint8_t digit = 0;

    if (d == 0) {
      // Gestion des zéros
      if (leading_zero || !leading || (k >= 6 - decimal - 1))
          digit = encodeDigit(d, false);
      else
          digit = 0;
    } else {
      // Gestion des autes valeurs
      digit = encodeDigit(d, false);
      num -= d * divisor;
      leading = false;
    }
    
    // Add the decimal point/colon to the digit
    digit |= (dots & 0x80); 
    dots <<= 1;
    
    digits[k] = digit;
  }

  setSegments(digits + (6 - m_nbDigit), m_nbDigit);
}

void TM1637Full::showTime(uint8_t hh, uint8_t mm, uint8_t ss) {
  uint8_t digits[6];
  bool leading = false;

  if (ss > 60) { digits[4] = 0; digits[5] = 0; leading = true;}
  else {
    digits[4] = encodeDigit(ss/10, false);
    digits[5] = encodeDigit(ss%10, false);
  }

  if (mm > 60) { digits[2] = 0; digits[3] = 0; leading = true;}
  else {
    digits[2] = encodeDigit(mm/10, false);
    digits[3] = encodeDigit(mm%10, (leading?false:true));
    leading = false;
  }

  if (hh > 24) { digits[0] = 0; digits[1] = 0; }
  else {
    digits[0] = encodeDigit(hh/10, false);
    digits[1] = encodeDigit(hh%10, (leading?false:true));
  }
  
  setSegments(digits, m_nbDigit);
}

void TM1637Full::bitDelay() {
  delayMicroseconds(100);
}

void TM1637Full::start() {
  pinMode(m_pinDIO, OUTPUT);
  bitDelay();
}

void TM1637Full::stop() {
  pinMode(m_pinDIO, OUTPUT);
  bitDelay();
  pinMode(m_pinClk, INPUT);
  bitDelay();
  pinMode(m_pinDIO, INPUT);
  bitDelay();
}

bool TM1637Full::writeByte(uint8_t b) {
  uint8_t data = b;

  // 8 Data Bits
  for(uint8_t i = 0; i < 8; i++) {
    // CLK low
    pinMode(m_pinClk, OUTPUT);
    bitDelay();

    // Set data bit
    if (data & 0x01)
      pinMode(m_pinDIO, INPUT);
    else
      pinMode(m_pinDIO, OUTPUT);
    bitDelay();

    // CLK high
    pinMode(m_pinClk, INPUT);
    bitDelay();
    
    data = data >> 1;
  }

  // Wait for acknowledge
  // CLK to zero
  pinMode(m_pinClk, OUTPUT);
  pinMode(m_pinDIO, INPUT);
  bitDelay();

  // CLK to high
  pinMode(m_pinClk, INPUT);
  bitDelay();
  
  uint8_t ack = digitalRead(m_pinDIO);
  if (ack == 0)
    pinMode(m_pinDIO, OUTPUT);
  bitDelay();

  pinMode(m_pinClk, OUTPUT);
  bitDelay();

  return ack;
}

uint8_t TM1637Full::readByte() {
  uint8_t data = 0x00;

  // 8 Data Bits
  pinMode(m_pinDIO, INPUT);  
  bitDelay();
    
  for(uint8_t i = 0; i < 8; i++) {
    data = data << 1;
    // CLK low
    pinMode(m_pinClk, OUTPUT);
    bitDelay();

    // CLK high
    pinMode(m_pinClk, INPUT);
    bitDelay();

    // Get data bit
    data |= digitalRead (m_pinDIO);
  }

  // Wait for acknowledge
  // CLK to zero
  pinMode(m_pinClk, OUTPUT);
  pinMode(m_pinDIO, INPUT);
  bitDelay();

  // CLK to high
  pinMode(m_pinClk, INPUT);
  bitDelay();
  uint8_t ack = digitalRead(m_pinDIO);
  if (ack == 0)
    pinMode(m_pinDIO, OUTPUT);
  bitDelay();

  // Stop
  pinMode(m_pinClk, OUTPUT);
  bitDelay();
  pinMode(m_pinClk, INPUT);
  bitDelay();
  pinMode(m_pinDIO, INPUT);
  bitDelay();

  return data;
}

uint8_t TM1637Full::encodeDigit(uint8_t digit, bool dot) {
  if (digit >= sizeof(digitToSegment)) return 0;
  uint8_t data = digitToSegment[digit];
  return data | (dot?0x80:0);
}

