/* ******************************** 
 *  library : TM1637Full.h
 *  
 *  Gestion d'un circuit TM1637 pour piloter un tube d'afficheurs 7 segments
 *  4 ou 6 digits ainsi que d'un clavier 6 touches.
 *  Projet réalisé sur les bases de la bibliothèque TM1637Display de <avishorp@gmail.com>
 *  
 *  @Author mrcjl <mjl1@morceau.fr>
 *  @Version 2.3 (23/12/2018)
 *  @see http://wikitechno.morceau.fr/doku.php?id=mbloc_disp4x7seg
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef __TM1637FULL__
#define __TM1637FULL__

#include <inttypes.h>

#define SEG_A   0b00000001
#define SEG_B   0b00000010
#define SEG_C   0b00000100
#define SEG_D   0b00001000
#define SEG_E   0b00010000
#define SEG_F   0b00100000
#define SEG_G   0b01000000

class TM1637Full {

public:
  /* Initialise l'object TM1637
   *  
   * @param pinClk - The number of the digital pin connected to the clock pin of the module
   * @param pinDIO - The number of the digital pin connected to the DIO pin of the module
   * @param nbDigits - Le nombre de digit de l'afficheur
   */
  TM1637Full(uint8_t pinDIO, uint8_t pinClk, uint8_t nbDigit = 4);

  /* Sets the brightness of the display.
   *
   * The setting takes effect when a command is given to change the data being displayed.
   * @param brightness A number from 0 (lowes brightness) to 7 (highest brightness)
   * @param on Turn display on or off
   */
  void setBrightness(uint8_t brightness, bool on = true);

  /* Display arbitrary data on the module
   *
   * This function receives raw segment values as input and displays them. The segment data
   * is given as a byte array, each byte corresponding to a single digit. Within each byte,
   * bit 0 is segment A, bit 1 is segment B etc.
   * The function may either set the entire display or any desirable part on its own. The first
   * digit is given by the @ref pos argument with 0 being the leftmost digit. The @ref length
   * argument is the number of digits to be set. Other digits are not affected.
   * @param segments An array of size @ref length containing the raw segment values
   * @param length The number of digits to be modified
   * @param pos The position from which to start the modification (0 - leftmost, 5 - rightmost)
   */
  void setSegments(const uint8_t segments[], uint8_t length = 4, uint8_t pos = 0);

  /* Affiche le code transmis pour chaque digit
   *
   * Converti chaque code transmis en valeur affichable (avec le point si négatif)
   * avant de les transmettre à l'afficheur
   * @param segments An array of size @ref length containing the raw segment values
   */
  void showRaw(const uint8_t segments[]);

  /* Affiche un nombre decimal
   *
   * Affiche un nomvre décimal avec les options suivantes
   * @param num Le nombre à afficher (0 à [99]9999)
   * @param leading_zero N'affiche pas les zéros non significatifs
   * @param decimal nombre de digit après la virgule (0 à 6)
   */
  void showNumberDec(long num, bool leading_zero = false, uint8_t decimal = 0);

  /* Affiche une heure
   *
   * Affiche une heure au format HH.MM.SS
   * @param hh heures
   * @param mm minutes
   * @param ss secondes
   */
  void showTime(uint8_t hh, uint8_t mm, uint8_t ss);

  /* Traduit un code pour l'afficher sur un digit 7 segments avec gestion du point
   *
   * La méthode accepte un code entre 0 et la taille du tableau "digitToSegment"
   * Toutes les autres possibilités sont remplacées par un digit vide.
   * On trouve les chiffres hexadécimaux suivit des rotations de digits et autres possibilités
   * @param digit un nombre entre 0 et sizeof(digitToSegment[])
   * @param dot L'affichage du point décimal
   * @return Le code representant la valeur à utiliser pour le 7 segment
   */
  uint8_t encodeDigit(uint8_t digit, bool dot);

  /* Retourne la position du bouton appuyé
   *
   * 0= pas de bouton ; les boutons vont de gauche 1 à droite 6
   * @return Un nombre entre 0 et 6 (ou 255 si anomalie)
   */
  uint8_t getBtn();

protected:
   void bitDelay();
   void start();
   void stop();
   bool writeByte(uint8_t b);
   uint8_t readByte();
   
private:
  uint8_t m_pinClk;
  uint8_t m_pinDIO;
  uint8_t m_brightness;
  uint8_t m_nbDigit;
};

#endif // __TM1637FULL__

